'use client';

import React, { useEffect, useState } from 'react';
import { useParams, useRouter } from 'next/navigation';
import { Calendar, Clock, MapPin, Users, Edit, Trash2, ArrowRight, User, Globe, Briefcase, Tag } from 'lucide-react';

interface Event {
    id: number;
    title: string;
    description?: string;
    start_datetime: string;
    end_datetime?: string;
    location?: string;
    event_type: string;
    status: string;
    customer_id?: string;
    customer_name?: string;
    customer_company?: string;
    customer_email?: string;
    customer_phone?: string;
    domain_name?: string;
    project_title?: string;
    attendees?: string;
    reminder_minutes?: number;
    tags?: string;
    created_at: string;
    updated_at: string;
}

export default function EventDetailsPage() {
    const params = useParams();
    const router = useRouter();
    const eventId = params.id;

    const [event, setEvent] = useState<Event | null>(null);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState<string | null>(null);
    const [deleting, setDeleting] = useState(false);

    useEffect(() => {
        loadEvent();
    }, [eventId]);

    const loadEvent = async () => {
        try {
            const response = await fetch(`/api/events/${eventId}`);
            if (response.ok) {
                const data = await response.json();
                setEvent(data);
            } else {
                setError('האירוע לא נמצא');
            }
        } catch (err) {
            setError('שגיאה בטעינת האירוע');
        } finally {
            setLoading(false);
        }
    };

    const deleteEvent = async () => {
        if (!confirm('האם אתה בטוח שברצונך למחוק אירוע זה?')) {
            return;
        }

        setDeleting(true);
        try {
            const response = await fetch(`/api/events/${eventId}`, {
                method: 'DELETE',
            });

            if (response.ok) {
                router.push('/events');
            } else {
                alert('שגיאה במחיקת האירוע');
            }
        } catch (err) {
            alert('שגיאה במחיקת האירוע');
        } finally {
            setDeleting(false);
        }
    };

    const formatDateTime = (datetime: string) => {
        const date = new Date(datetime);
        return {
            date: date.toLocaleDateString('he-IL', {
                weekday: 'long',
                year: 'numeric',
                month: 'long',
                day: 'numeric'
            }),
            time: date.toLocaleTimeString('he-IL', { hour: '2-digit', minute: '2-digit' })
        };
    };

    const getEventTypeIcon = (type: string) => {
        const icons: Record<string, string> = {
            meeting: '🤝',
            call: '📞',
            presentation: '📊',
            webinar: '💻',
            deadline: '⏰',
            reminder: '🔔',
            other: '📌'
        };
        return icons[type] || '📌';
    };

    const getEventTypeLabel = (type: string) => {
        const labels: Record<string, string> = {
            meeting: 'פגישה',
            call: 'שיחה',
            presentation: 'מצגת',
            webinar: 'וובינר',
            deadline: 'דדליין',
            reminder: 'תזכורת',
            other: 'אחר'
        };
        return labels[type] || type;
    };

    const getStatusColor = (status: string) => {
        const colors: Record<string, string> = {
            scheduled: 'bg-blue-100 text-blue-800',
            in_progress: 'bg-yellow-100 text-yellow-800',
            completed: 'bg-green-100 text-green-800',
            cancelled: 'bg-red-100 text-red-800',
            postponed: 'bg-gray-100 text-gray-800'
        };
        return colors[status] || 'bg-gray-100 text-gray-800';
    };

    const getStatusLabel = (status: string) => {
        const labels: Record<string, string> = {
            scheduled: 'מתוכנן',
            in_progress: 'בתהליך',
            completed: 'הושלם',
            cancelled: 'בוטל',
            postponed: 'נדחה'
        };
        return labels[status] || status;
    };

    if (loading) {
        return (
            <div className="min-h-screen bg-gray-50 p-8" dir="rtl">
                <div className="max-w-4xl mx-auto">
                    <div className="bg-white shadow rounded-lg p-8">
                        <div className="text-center text-gray-500">טוען...</div>
                    </div>
                </div>
            </div>
        );
    }

    if (error || !event) {
        return (
            <div className="min-h-screen bg-gray-50 p-8" dir="rtl">
                <div className="max-w-4xl mx-auto">
                    <div className="bg-white shadow rounded-lg p-8">
                        <div className="text-center text-red-600 mb-4">{error || 'האירוע לא נמצא'}</div>
                        <div className="text-center">
                            <a
                                href="/events"
                                className="text-blue-600 hover:text-blue-800"
                            >
                                חזרה לרשימת אירועים
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        );
    }

    const startTime = formatDateTime(event.start_datetime);
    const endTime = event.end_datetime ? formatDateTime(event.end_datetime) : null;
    const attendees = event.attendees ? JSON.parse(event.attendees) : [];
    const tags = event.tags ? JSON.parse(event.tags) : [];

    return (
        <div className="min-h-screen bg-gray-50 p-8" dir="rtl">
            <div className="max-w-4xl mx-auto">
                {/* Header */}
                <div className="mb-6">
                    <a
                        href="/events"
                        className="inline-flex items-center gap-2 text-blue-600 hover:text-blue-800 mb-4"
                    >
                        <ArrowRight className="w-4 h-4" />
                        חזרה לרשימת אירועים
                    </a>
                </div>

                {/* Event Details */}
                <div className="bg-white shadow rounded-lg overflow-hidden">
                    {/* Title Section */}
                    <div className="p-6 border-b border-gray-200">
                        <div className="flex items-start justify-between">
                            <div className="flex-1">
                                <div className="flex items-center gap-3 mb-3">
                                    <span className="text-3xl">
                                        {getEventTypeIcon(event.event_type)}
                                    </span>
                                    <div>
                                        <h1 className="text-2xl font-bold text-gray-900">
                                            {event.title}
                                        </h1>
                                        <div className="flex items-center gap-2 mt-2">
                                            <span className={`inline-flex items-center px-3 py-1 rounded-full text-sm ${getStatusColor(event.status)}`}>
                                                {getStatusLabel(event.status)}
                                            </span>
                                            <span className="text-sm text-gray-500">
                                                {getEventTypeLabel(event.event_type)}
                                            </span>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <div className="flex items-center gap-2">
                                <a
                                    href={`/events/${event.id}/edit`}
                                    className="px-4 py-2 text-blue-600 hover:bg-blue-50 rounded-md transition-colors flex items-center gap-2"
                                >
                                    <Edit className="w-4 h-4" />
                                    ערוך
                                </a>
                                <button
                                    onClick={deleteEvent}
                                    disabled={deleting}
                                    className="px-4 py-2 text-red-600 hover:bg-red-50 rounded-md transition-colors flex items-center gap-2 disabled:opacity-50"
                                >
                                    <Trash2 className="w-4 h-4" />
                                    מחק
                                </button>
                            </div>
                        </div>

                        {event.description && (
                            <div className="mt-4 text-gray-700" dangerouslySetInnerHTML={{ __html: event.description }} />
                        )}
                    </div>

                    {/* Date & Time */}
                    <div className="p-6 border-b border-gray-200">
                        <h2 className="text-lg font-semibold text-gray-900 mb-4 flex items-center gap-2">
                            <Calendar className="w-5 h-5" />
                            תאריך ושעה
                        </h2>
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                            <div>
                                <div className="text-sm text-gray-500 mb-1">התחלה</div>
                                <div className="text-gray-900">
                                    <div>{startTime.date}</div>
                                    <div className="flex items-center gap-2 text-sm text-gray-600 mt-1">
                                        <Clock className="w-4 h-4" />
                                        {startTime.time}
                                    </div>
                                </div>
                            </div>
                            {endTime && (
                                <div>
                                    <div className="text-sm text-gray-500 mb-1">סיום</div>
                                    <div className="text-gray-900">
                                        <div>{endTime.date}</div>
                                        <div className="flex items-center gap-2 text-sm text-gray-600 mt-1">
                                            <Clock className="w-4 h-4" />
                                            {endTime.time}
                                        </div>
                                    </div>
                                </div>
                            )}
                        </div>
                    </div>

                    {/* Location */}
                    {event.location && (
                        <div className="p-6 border-b border-gray-200">
                            <h2 className="text-lg font-semibold text-gray-900 mb-4 flex items-center gap-2">
                                <MapPin className="w-5 h-5" />
                                מיקום
                            </h2>
                            <div className="text-gray-900">{event.location}</div>
                        </div>
                    )}

                    {/* Attendees */}
                    {attendees.length > 0 && (
                        <div className="p-6 border-b border-gray-200">
                            <h2 className="text-lg font-semibold text-gray-900 mb-4 flex items-center gap-2">
                                <Users className="w-5 h-5" />
                                משתתפים ({attendees.length})
                            </h2>
                            <ul className="space-y-2">
                                {attendees.map((attendee: string, index: number) => (
                                    <li key={index} className="flex items-center gap-2 text-gray-900">
                                        <User className="w-4 h-4 text-gray-400" />
                                        {attendee}
                                    </li>
                                ))}
                            </ul>
                        </div>
                    )}

                    {/* Related Items */}
                    {(event.customer_name || event.domain_name || event.project_title) && (
                        <div className="p-6 border-b border-gray-200">
                            <h2 className="text-lg font-semibold text-gray-900 mb-4">קשרים</h2>
                            <div className="space-y-3">
                                {event.customer_name && (
                                    <div className="flex items-center gap-3">
                                        <User className="w-5 h-5 text-gray-400" />
                                        <div>
                                            <div className="text-sm text-gray-500">לקוח</div>
                                            <div className="text-gray-900">{event.customer_name}</div>
                                            {event.customer_company && (
                                                <div className="text-sm text-gray-600">{event.customer_company}</div>
                                            )}
                                        </div>
                                    </div>
                                )}
                                {event.domain_name && (
                                    <div className="flex items-center gap-3">
                                        <Globe className="w-5 h-5 text-gray-400" />
                                        <div>
                                            <div className="text-sm text-gray-500">דומיין</div>
                                            <div className="text-gray-900">{event.domain_name}</div>
                                        </div>
                                    </div>
                                )}
                                {event.project_title && (
                                    <div className="flex items-center gap-3">
                                        <Briefcase className="w-5 h-5 text-gray-400" />
                                        <div>
                                            <div className="text-sm text-gray-500">פרויקט</div>
                                            <div className="text-gray-900">{event.project_title}</div>
                                        </div>
                                    </div>
                                )}
                            </div>
                        </div>
                    )}

                    {/* Tags */}
                    {tags.length > 0 && (
                        <div className="p-6 border-b border-gray-200">
                            <h2 className="text-lg font-semibold text-gray-900 mb-4 flex items-center gap-2">
                                <Tag className="w-5 h-5" />
                                תגיות
                            </h2>
                            <div className="flex flex-wrap gap-2">
                                {tags.map((tag: any, index: number) => (
                                    <span
                                        key={index}
                                        className="inline-flex items-center px-3 py-1 rounded-full text-sm"
                                        style={{
                                            backgroundColor: tag.color + '20',
                                            color: tag.color
                                        }}
                                    >
                                        {tag.name}
                                    </span>
                                ))}
                            </div>
                        </div>
                    )}

                    {/* Reminder */}
                    {event.reminder_minutes !== undefined && event.reminder_minutes > 0 && (
                        <div className="p-6 border-b border-gray-200">
                            <h2 className="text-lg font-semibold text-gray-900 mb-4">תזכורת</h2>
                            <div className="text-gray-900">
                                {event.reminder_minutes < 60
                                    ? `${event.reminder_minutes} דקות לפני`
                                    : event.reminder_minutes < 1440
                                        ? `${Math.floor(event.reminder_minutes / 60)} שעות לפני`
                                        : `${Math.floor(event.reminder_minutes / 1440)} ימים לפני`
                                }
                            </div>
                        </div>
                    )}

                    {/* Metadata */}
                    <div className="p-6 bg-gray-50">
                        <div className="text-xs text-gray-500 space-y-1">
                            <div>נוצר: {new Date(event.created_at).toLocaleString('he-IL')}</div>
                            <div>עודכן: {new Date(event.updated_at).toLocaleString('he-IL')}</div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    );
}

